﻿using log4net;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Schema;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.Functions.Configuration.Interface;
using VA.PPMS.IWS.SchemaValidationService.Interface;
using VA.PPMS.ProviderData;
using static VA.PPMS.IWS.Functions.Configuration.Interface.SchemaOptions;

namespace VA.PPMS.IWS.SchemaValidationService
{
    public class SchemaValidationService : ISchemaValidationService
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;

        public SchemaValidationService(ILog logger, IIwsConfiguration configuration)
        {
            _logger = logger;
            _configuration = configuration;
        }

        public async Task<List<Validation>> ValidateSchemaAsync(Providers providers)
        {
            string id = null;
            var returnValue = new List<Validation>();

            try
            {
                _logger.Info($"@@@@ INFO - Start ValidateSchemaAsync @@@@");

                var schemaOption = providers.IsVaNetwork ? SchemaProfiles.VA : SchemaProfiles.CCN;
                var packet = await _configuration.GetSchemaProfileAsync(schemaOption);
                
                var providerCollectionResult = new ConcurrentDictionary<string, Validation>();

                var suffix = 1;

                foreach (var provider in providers.Provider)
                {
                    // If insert, validate against schema
                    var validationResult = provider.IsModification ? string.Empty : ValidateSchema(provider, packet);
                    id = provider.ProviderId;
                    
                    // Check to make sure ProviderId is unique in batch
                    if (providerCollectionResult.ContainsKey(id))
                    {
                        id = $"{id}-{suffix++}";
                        validationResult = "Duplicate ProviderId";
                    }

                    providerCollectionResult.TryAdd(id, new Validation(provider.ProviderId, provider.CorrelationId, provider.ProviderNameDisplay, (int)provider.TransactionType, validationResult));
                }

                foreach (var validation in providerCollectionResult)
                {
                    returnValue.Add(validation.Value);
                }

                providerCollectionResult.Clear();

                _logger.Info($"@@@@ INFO - End ValidateSchemaAsync @@@@");
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ ERROR - There was a problem with the Schema Validation Service @@@@", ex);
                returnValue.Add(GetErrorValidationResult(id, $"There was a problem with the Schema Validation Service. {ex.Message}"));
            }

            return returnValue;
        }

        /// <summary>
        /// Validate xml against schema. Return any errors found.
        /// </summary>
        /// <typeparam name="T">Object model type</typeparam>
        /// <param name="classInstance">Object model</param>
        /// <param name="profile">Schema profile</param>
        /// <returns>Error messages encountered</returns>
        private static string ValidateSchema<T>(T classInstance, SchemaProfile profile)
        {
            var sb = new StringBuilder();
            try
            {
                var payload = Utilities.SerializeInstance(classInstance, profile.Prefix, profile.Namespace);

                var customDataFilePath = $"{profile.SchemaName}";

                var xmlReaderSettings = new XmlReaderSettings();
                xmlReaderSettings.Schemas.Add(profile.Namespace, customDataFilePath);
                xmlReaderSettings.ValidationType = ValidationType.Schema;
                xmlReaderSettings.ValidationEventHandler += (o, e) => { if (e.Severity == XmlSeverityType.Error) sb.Append(e.Message); };

                var reader = XmlReader.Create(new StringReader(payload), xmlReaderSettings);

                while (reader.Read()) { }
            }
            catch (Exception ex)
            {
                sb.Append(ex.Message);
            }

            return sb.ToString();
        }

        private Validation GetErrorValidationResult(string id, string message)
        {
             return new Validation(id, null, null, 0, message);
        }
    }
}